/** @file   sentrygun.cpp
 * @brief   Implementation of SentryGun class
 * @version $Revision: 1.3 $
 * @date    $Date: 2006/06/21 16:08:38 $
 * @author  Tomi Lamminsaari
 */

#include "sentrygun.h"
#include "sentrygunai.h"
#include "GfxManager.h"
#include "www_map.h"
#include "soundsamples.h"
#include "gameanims.h"
#include "animplayer.h"
#include "redrawqueue.h"
#include "warglobals.h"
#include "gfxid.h"
#include "AnimId.h"
#include "Settings.h"
using namespace eng2d;

namespace WeWantWar {


//********************************************************************
//                                                                   *
//      Static members and constants                                 *
//                                                                   *
//********************************************************************

const int SentryGun::GUN_CTRLPOINT_INDEX;
const int SentryGun::FIREBALL_CTRLPOINT1_INDEX;
const int SentryGun::FIREBALL_CTRLPOINT2_INDEX;
const int SentryGun::MACHINEGUN_RELOAD_DELAY;
const int SentryGun::FIREBALL_RELOAD_DELAY;


//********************************************************************
//                                                                   *
//      Constructors, destructor and operators                       *
//                                                                   *
//********************************************************************

/** Constructor.
 */
SentryGun::SentryGun( GunType t ) :
  Alien(),
  m_gunType( t )
{
  // Set the properties
  if ( t == T_MACHINEGUN ) {
    this->setArmor( 3.5 );
    this->boundingSphere( 30 );
    
  } else if ( t == T_FIREBALL ) {
    this->setArmor( 16.0 );
    this->boundingSphere( 40 );
    
  }
  
  // Create controlpoint for gun's nose
  this->addCtrlPoint( eng2d::Vec2D(0, -26) );
  this->addCtrlPoint( eng2d::Vec2D(-19, -38 ) );
  this->addCtrlPoint( eng2d::Vec2D( 19, -38 ) );
  
  this->setProperties( 0 );
  
  // And create the controller.
  this->setController( new SentryGunAI(this) );
}



/** Destructor.
 */
SentryGun::~SentryGun()
{
}



//********************************************************************
//                                                                   *
//      Public interface                                             *
//                                                                   *
//********************************************************************

/** Updates this object.
 */
void SentryGun::update()
{
  if ( this->state() == GameObject::STATE_KILLED ) {
    return;
  }
  
  switch ( m_gunType ) {
    case ( T_MACHINEGUN ): {
      this->updateMachinegun();
      break;
    }
    case ( T_FIREBALL ): {
      this->updateFireballgun();
      break;
    }
  }
  
}



/** Redraws this object.
 */
void SentryGun::redraw( RedrawQueue* pQueue )
{

  if ( this->state() == GameObject::STATE_KILLED ) {
    return;
  }
  
  RLE_SPRITE* sentrySupport =
      GfxManager::findRleSprite(GfxId::KAlienSentrySupport, 0);
  BITMAP* sentryGun = 0;
  
  switch ( m_gunType ) {
    case ( T_MACHINEGUN ): {
      sentryGun = GfxManager::findBitmap(GfxId::KAlienSentryGun1, 0);
      break;
    }
    case ( T_FIREBALL ): {
      sentryGun = GfxManager::findBitmap(GfxId::KAlienSentryGun3, 0);
      break;
    }
    default: {
      sentryGun = 0;
      break;
    }
  }
  eng2d::Vec2D p( this->position() );
  int x = static_cast<int>( p.x() ) - Map::scrollX;
  int y = static_cast<int>( p.y() ) - Map::scrollY;
  x -= sentryGun->w / 2;
  y -= sentryGun->h / 2;
  
  if ( m_gunType == T_MACHINEGUN ) {
    pQueue->add( RedrawQueue::PRI_BELOW_NORMAL, x,y, RedrawQueue::SPTYPE_RLE,
                 sentrySupport );
  }
  pQueue->addRotatedSprite( RedrawQueue::PRI_NORMAL, x,y,
                            itofix( this->angle() ), sentryGun );
}



/** Makes sound
 */
void SentryGun::makeSound( GameObject::SoundID id ) const
{
  if ( id == GameObject::SND_ATTACK ) {
    switch ( this->gunType() ) {
      case ( T_MACHINEGUN ): {
        Weapon w( Weapon::W_MINIGUN );
        Sound::playSample( w.getSoundID(), false );
        break;
      }
      case ( T_FLAMETHROWER ): {
        Weapon w( Weapon::W_FLAMETHROWER );
        Sound::playSample( w.getSoundID(), false );
        break;
      }
      case ( T_FIREBALL ): {
        Sound::playSample( SMP_SENTRYSHOOT3, false );
        break;
      }
    }
  } else if ( id == GameObject::SND_PAIN ) {
    Sound::playSample( SMP_METALHIT, false );
    
  }
}



/** Kills this object.
 */
void SentryGun::kill()
{
  this->state( GameObject::STATE_KILLED );
  this->hidden( true );
  
  Sound::playSample(SMP_GRENADE, false);
  const Animation& lightAnim = GameAnims::findAnimation( AnimId::KGrenadeExplosionLight );
  if ( Settings::explosionLights == true ) {
    AnimPlayer::spawn( lightAnim, this->position(), 0 );
  }
  const Animation& lightRing = GameAnims::findAnimation( AnimId::KExplosionLightRing );
  AnimPlayer::spawn( lightRing, this->position() );
  const Animation& explosionAnim =
      GameAnims::findAnimation( AnimId::KExplosionGrenade, 0 );
  AnimPlayer::spawn( explosionAnim, this->position(), 0 );

  // And as the sentrygun explodes, it emits some bullets
  float ac = 256.0 / 16.0;
  for (float a=0; a < 255; a+=ac) {
    eng2d::Vec2D spdVec( 0,-7 );
    spdVec.rotate( a );
    
    Bullet* b = BulletTable::createBullet( this, m_position, Bullet::ERifle );
    b->setVelocity( spdVec );
    b->iRange = 110;
    b->iDamage = 10;
    
    WarGlobals::pBulletManager->spawnBullet( b );
  }
}



/** Handles the bullethits
 */
bool SentryGun::hitByBullet( Bullet* pB )
{
  bool ret = Alien::hitByBullet( pB );
  if ( ret == true ) {
    ParticleSparks* pP = new ParticleSparks( pB->iPosition, pB->velocity(), 12 );
    WarGlobals::pPartManager->addSystem( pP );
  }
  return ret;
}



//********************************************************************
//                                                                   *
//      Public GET - methods                                         *
//                                                                   *
//********************************************************************

/** Returns the type of this object.
 */
ObjectID::Type SentryGun::objectType() const
{
  return ObjectID::TYPE_SENTRYGUN;
}



/** Returns the guntype
 */
SentryGun::GunType SentryGun::gunType() const
{
  return m_gunType;
}



//********************************************************************
//                                                                   *
//      Private methods                                              *
//                                                                   *
//********************************************************************

/** Updates the machinegun
 */
void SentryGun::updateMachinegun()
{
  // Update the controller.
  BaseController* pC = this->getController();
  pC->update();
  
  // Rotate us if necessary
  this->changeAngle( pC->turn() );

  // Shoot if necessary
  if ( pC->shoot() != 0 ) {
    this->attack();
  }
}



/** Updates the fireballgun
 */
void SentryGun::updateFireballgun()
{

  // Update the controller.
  BaseController* pC = this->getController();
  pC->update();

  // Rotate us if necessary
  this->changeAngle( pC->turn() );

  // Shoot if necessary
  if ( pC->shoot() != 0 ) {
    this->attack();
  }
}



/** Sets the correct animation.
 */
void SentryGun::setCorrectAnimation( int aAnimId )
{
}



/** Shoots
 */
void SentryGun::attack()
{
  if ( this->reloading() == false ) {
    switch ( m_gunType ) {
      case ( T_MACHINEGUN ): {
        this->shootMachineGun();
        break;
      }
      case ( T_FIREBALL ): {
        this->shootFireball();
        break;
      }
    }
  }
}



/** Shoots with machine gun
 */
void SentryGun::shootMachineGun()
{
  eng2d::Vec2D gunV( this->getCtrlPoint(GUN_CTRLPOINT_INDEX) );
  gunV += this->position();
    
  Bullet* pB = BulletTable::createBullet( this, gunV, Bullet::EMinigun );
  pB->iDamage = 15;
  WarGlobals::pBulletManager->spawnBullet( pB );

  this->makeSound( GameObject::SND_ATTACK );
  this->setCounter( RELOAD_COUNTER_INDEX, MACHINEGUN_RELOAD_DELAY );

  // Create the shooting animations
  const Animation& flameAnim = GameAnims::findAnimation( AnimId::KMinigunShootFlame );
  const Animation& lightAnim = GameAnims::findAnimation( AnimId::KRifleShotLight );
  AnimPlayer::spawn( flameAnim, gunV, 0 );
  if ( Settings::shootingLights == true ) {
    AnimPlayer::spawn( lightAnim, gunV, 0 );
  }
}



/** Shoots with fireballs
 */
void SentryGun::shootFireball()
{
  eng2d::Vec2D gunV1( this->getCtrlPoint(FIREBALL_CTRLPOINT1_INDEX) );
  eng2d::Vec2D gunV2( this->getCtrlPoint(FIREBALL_CTRLPOINT2_INDEX) );

  gunV1 += this->position();
  gunV2 += this->position();

  
  Bullet* pB1 = BulletTable::createBullet( this, gunV1, Bullet::ESentryFireball );
  Bullet* pB2 = BulletTable::createBullet( this, gunV2, Bullet::ESentryFireball );

  WarGlobals::pBulletManager->spawnBullet( pB1 );
  WarGlobals::pBulletManager->spawnBullet( pB2 );

  this->makeSound( GameObject::SND_ATTACK );
  this->setCounter( RELOAD_COUNTER_INDEX,   FIREBALL_RELOAD_DELAY );

  // Create the shooting animations
  const Animation& flameAnim = GameAnims::findAnimation( AnimId::KRifleShootFlame );
  const Animation& lightAnim = GameAnims::findAnimation( AnimId::KRifleShotLight );
  AnimPlayer::spawn( flameAnim, gunV1, 0 );
  AnimPlayer::spawn( flameAnim, gunV2, 0 );
  if ( Settings::shootingLights == true ) {
    AnimPlayer::spawn( lightAnim, gunV1, 0 );
    AnimPlayer::spawn( lightAnim, gunV2, 0 );
  }
}


} // end of namespace

